﻿using System.Collections;
using System.Collections.Generic;
using TMPro;
using UnityEditor;
using UnityEngine;
using UnityEngine.EventSystems;
using UnityEngine.UI;

public class RoleSelect : MonoBehaviour
{
    public GameObject PlayerOne;    //Parent game object of Player One's controls
    public GameObject PlayerTwo;    //Parent game object of Player Two's controls
    public PageNavigator ForwardNavigator;

    private RoleSelectState _CurrentState;  //The current state of the role select UI

    /// <summary>
    /// Checks the selected roles to make sure there isn't a conflict and then moves onto the next page
    /// </summary>
    public void CheckRoles()
    {
        TextItemsSelector playerOneTextSelector = PlayerOne.FindChild("Content").FindChild("RoleSelector").GetComponent<TextItemsSelector>();
        TextItemsSelector playerTwoTextSelector = PlayerTwo.FindChild("Content").FindChild("RoleSelector").GetComponent<TextItemsSelector>();

        if (playerOneTextSelector != null && playerTwoTextSelector != null)
        {
            //We've got the controls, check the selected indexes
            if (playerOneTextSelector.CurrentIndex != playerTwoTextSelector.CurrentIndex)
            {
                //Different indexes so different roles and no conflict, set the roles in the game controller
                if (playerOneTextSelector.SelectedItemText.ToLower() == "hider")
                {
                    GameController.PlayerOneRole = Constants.PlayerRole.Hider;
                    GameController.PlayerTwoRole = Constants.PlayerRole.Seeker;
                }

                else
                {
                    GameController.PlayerOneRole = Constants.PlayerRole.Seeker;
                    GameController.PlayerTwoRole = Constants.PlayerRole.Hider;
                }

                //Get the sprite controls and set the sprites in the game controller
                SpriteItemsSelector playerOneSpriteSelector = PlayerOne.FindChild("Content").FindChild("CharacterSelector").GetComponent<SpriteItemsSelector>();
                SpriteItemsSelector playerTwoSpriteSelector = PlayerTwo.FindChild("Content").FindChild("CharacterSelector").GetComponent<SpriteItemsSelector>();
                GameController.PlayerOneSprite = playerOneSpriteSelector.SelectedSprite;
                GameController.PlayerTwoSprite = playerTwoSpriteSelector.SelectedSprite;

                ForwardNavigator.Navigate();    //All setup, now navigate forward
            }

            else
            {
                //Indexes match so there's a conflict
                CustomMessageBox errorBox = new CustomMessageBox();
                MessageBoxManager.Instance.Show("Role Conflict", "Please ensure both players pick different roles!", new List<string>() { "OK" });
            }
        }
    }

    public void OnNextButtonPressed()
    {
        if (!PageManager.IgnoreBPress && !MessageBoxManager.Instance.ShowingMessageBox)
        {
            _CurrentState.MoveToNextState();    //Move onto the next UI state
        }
    }

    public void OnBackButtonPressed()
    {
        if (!PageManager.IgnoreBPress && !MessageBoxManager.Instance.ShowingMessageBox)
        {
            _CurrentState.MoveToPreviousState();    //Move to the previous UI state
        }
    }

    public void Start()
    {
        //Starting, so we begin at the player one UI state
        RoleSelectState.RoleSelect = this;
        ChangeCurrentState(new PlayerOneState());
    }

    /// <summary>
    /// Changes from the current UI state to the new UI state, internally handling calls to OnEnd and OnStart
    /// </summary>
    /// <param name="newState">The new UI state to become current</param>
    public void ChangeCurrentState(RoleSelectState newState)
    {
        if(_CurrentState != null)
        {
            //We've got a current phase, end it
            _CurrentState.OnEndCurrentState();
        }

        //Start the new state and set it to be current
        _CurrentState = newState;
        _CurrentState.OnStartCurrentState();
    }
}
